/****** Object:  StoredProcedure [dbo].[RP_UserEligibility]    Script Date: 1/7/2015 11:05:03 AM ******/
SET ANSI_NULLS ON
GO

SET QUOTED_IDENTIFIER ON
GO



/* =============================================
-- Author:		<Dave Hartless>
-- Create date: <2013-07-27>
-- Description:	<Process the source RewardSplit table into an all inclusive list of users AND owners AND
--				 their defined percentages of the reward split.>
--
 Process Steps:
 1.) compute owner splits from postgress and insert them into staging table
 2.) get elerium project members
 3.) Remove any duplicate records in source table that are duplicate for projectID AND userID
 4.) Remove all project records in source table WHERE the project was blacklisted
	a.) Projects can only have a status of N AND Stage can not be X or D
	b.) remove all projects where the owner is not in the rewards program
	c.) now delete from staging table all users that are NOT eligible, have to first get ones that have no entry in reward user via a not in
 5.) unintentially left blank
 6.) Process all users in source table who have 100% allocated AND are determined to be eligible.
	a.) Remove all records FROM source table WHERE project ID exist FROM inserts FROM step 6
 7.) Find all users in source table who have 100% allocated AND are determined NOT to be eligible, DELETE them	
	a.) now get the left over owners at zero in staging for the projects that made it into RewardSplit		
 
 9.) Process all users in source table who have between 1% AND 99% allocated, at this point after step 5 removed all NON eligible user there only eligible ones left
 8.) Delete any user that is not eligible and has less than 100% split			
 10.) Process all users in source table who have 0% allocated
CHANGE LOG:
James S Davis <2015-01-12> Complete re vamp. 
-- ============================================= */

ALTER PROCEDURE [dbo].[RP_UserEligibility]
	@errorMessage varchar(2048) OUTPUT
AS
BEGIN
	
SET XACT_ABORT ON;

DECLARE @projectid int

DECLARE @projectIDStep9 int
DECLARE @eligUsersStep9 int
DECLARE @splitToDistroStep9 decimal(3,2)
DECLARE @splitToAddToUser decimal(3,2)
DECLARE @remainderStep9 decimal(3,2)
DECLARE @projectDistroCheckID int
DECLARE @userIdStep9 int
DECLARE @userGivenPercentageStep6 DECIMAL(3,2)
DECLARE @percentToDistro decimal(3,2)
DECLARE @percentToDistroINT int
DECLARE @splitToAddToUserINT int
DECLARE @remainderStep9INT int
DECLARE @isOwner bit
DECLARE @remainderGiven bit = 0
DECLARE @countOwnerInTableTwice int

DECLARE @eligibilityDateRange Date = GETUTCDATE() - 60;
-- Change on 2014-01-16, James Davis, now any time the @eligibilityDateRange is checked it is not only checked against LastLoginOnLegacyCF (last login to python curseforge)
-- but it is checked against [dbo].[UserProfile].DateLastVisit last visit to Elerium curseforge
SELECT @errorMessage = 'completed sucessfully';
BEGIN TRY

BEGIN TRANSACTION;

-- 1) compute owner splits from postgress and insert them into staging table
INSERT INTO [dbo].[RewardSplit_Staging] (Percentage, ProjectID, UserID)
SELECT 1.0 - sum(rss.Percentage) AS Percentage, rss.ProjectID, rlp.OwnerID
  FROM [dbo].[RewardSplit_Staging] rss with (nolock)
  LEFT JOIN [dbo].[RewardLegacyProject] rlp ON rss.ProjectID = rlp.ID
  WHERE rss.UserID != rlp.OwnerID
  GROUP BY rss.ProjectID, rlp.OwnerID;

/* STEP 2 Have to get elerium project members that have a percentage 
or that are the owner with zero percent so they can be included in getting distro of non eligible redistribution*/
;WITH EleriumSplits (ProjectID, UserID, Percentage) AS
	(
		SELECT pm.ProjectID, pm.UserID, pm.RewardPercentage
		FROM [dbo].[ProjectMember] pm with (nolock)
		LEFT JOIN [dbo].[Project] p ON pm.ProjectID = p.ID 
		WHERE pm.RewardPercentage > 0 OR (pm.RewardPercentage = 0 AND pm.UserID = p.OwnerID)	
	)
	MERGE [dbo].[RewardSplit_Staging] AS TARGET
	USING EleriumSplits AS SOURCE
	ON (TARGET.ProjectID = SOURCE.ProjectID AND TARGET.UserID = SOURCE.UserID)
	WHEN NOT MATCHED BY TARGET THEN
		INSERT(ProjectID, UserID, Percentage) VALUES(SOURCE.ProjectID, SOURCE.UserID, SOURCE.Percentage)
	WHEN MATCHED THEN
		UPDATE SET TARGET.Percentage = SOURCE.Percentage;

/* STEP 3 */
-- 3.) Remove any duplicate records in source table that are duplicate for projectID AND userID
--	There was an example in the source data WHERE same user was listed for same project multiple times. 
WITH numbered
	AS ( SELECT projectid, userid, row_number() OVER ( PARTITION BY projectid, userid ORDER BY projectid, userid ) AS nr
		FROM [dbo].[RewardSplit_Staging] with (nolock))
    DELETE  FROM numbered
    WHERE nr > 1;

/* STEP 4 */
-- 4.) Remove all project records in source table WHERE the project is blacklisted
-- If a project is blacklisted, then no one gets points, so remove all records for that project FROM source
-- table AS we will not process any records for it
DELETE  rss
 FROM [dbo].[RewardSplit_Staging] AS rss 
JOIN [dbo].[RewardProject] AS rp ON rss.ProjectID = rp.id
where IsBlacklisted = 1

/* STEP 4 a). */
-- 4a.) Projects can only have a status of N AND Stage can not be X or D
DELETE rss FROM [dbo].[RewardSplit_Staging] rss 
WHERE rss.ProjectID NOT IN (SELECT rss.ProjectID
	  FROM [dbo].[RewardSplit_Staging] rss
	  JOIN [dbo].[RewardLegacyProject] rlp ON rss.ProjectID = rlp.ID
	  WHERE rlp.Status = 'n' 
	  AND rlp.Stage NOT IN ('x','d'))

--4.)b.) remove all projects where the owner is not in the rewards program  MIGHT DO THIS ELSE WHERE
DELETE rss FROM [dbo].[RewardSplit_Staging] rss 
  left join [dbo].[RewardLegacyProject] rlp ON rss.ProjectID = rlp.id
WHERE rlp.OwnerID NOT IN (SELECT UserID FROM [dbo].[RewardUser] WHERE IsBannedOnLegacyCF = 0 and IsEnrolled = 1 and IsBlacklisted = 0)

-- 4c.) now delete from staging table all users that are NOT eligible, have to first get ones that have no entry in reward user via a not in
DELETE rss
	FROM [dbo].[RewardSplit_Staging] AS rss
	WHERE UserID NOT IN (SELECT UserID FROM [dbo].[RewardUser])

/*Create a temptable that is a combo of the userprofile and RewardUser table for last login, this way nulls in both are pre handeled  */
IF object_id('tempdb..##usersLastLogin') is not null
    DROP TABLE #usersLastLogin

CREATE TABLE #usersLastLogin(userID int , lastLogin datetime);
INSERT INTO #usersLastLogin (userID, lastLogin)
SELECT [UserID], DateLastVisit
  FROM [dbo].[UserProfile];

INSERT INTO #usersLastLogin (userID, lastLogin)
SELECT [UserID], LastLoginOnLegacyCF
  FROM [dbo].[RewardUser]

/* STEP 6 */
-- 6.) Process all users in source table who have 100% allocated AND are determined to be eligible.
-- If a user has 100% allocated AND they are eligible, insert them directly into the new table
--INSERT INTO RewardSplit
INSERT INTO [dbo].[RewardSplit] (projectID, userID, percentage, DailyBudgetDate) 
SELECT rss.ProjectID, rss.UserID, rss.Percentage, GETUTCDATE()
FROM [dbo].[RewardSplit_Staging] AS rss with (nolock)
LEFT JOIN [dbo].[RewardUser] AS rlu ON rss.UserID = rlu.UserID
WHERE rss.Percentage = 1.00 
AND rlu.IsBannedOnLegacyCF = 0 
AND rlu.IsBlacklisted = 0 
AND rlu.IsEnrolled = 1 
AND (select max(up.lastLogin) from #usersLastLogin up where up.userid = rlu.userid) >= @eligibilityDateRange;


-- 6.) a.) Now, remove all records FROM source table WHERE the projectID matches the projectIDs we just processed
	DELETE rss
	FROM [dbo].[RewardSplit_Staging] AS rss 
	LEFT JOIN [dbo].[RewardUser] AS rlu ON rss.UserID = rlu.UserID
	WHERE rss.Percentage = 1.00 
	AND rlu.IsBannedOnLegacyCF = 0 
	AND rlu.IsBlacklisted = 0
	AND rlu.IsEnrolled = 1 
	AND (select max(up.lastLogin) from #usersLastLogin up where up.userid = rlu.userid) >= @eligibilityDateRange;

-- 6.) b.) now get the zero owner rows that where inserted just in case the user getting 100% was not eligible
	DELETE rss
	FROM [dbo].[RewardSplit_Staging] AS rss 
	WHERE rss.ProjectID in (SELECT ProjectID FROM [dbo].[RewardSplit] with (nolock) WHERE Percentage = 1) AND rss.Percentage = 0

  /* STEP 7 */
  -- 7.) Find all users in source table who have 100% allocated AND are determined NOT to be eligible, DELETE them  
DELETE rss
FROM [dbo].[RewardSplit_Staging] AS rss  
LEFT JOIN [dbo].[RewardUser] AS rlu ON rss.UserID = rlu.UserID
WHERE  
	(rlu.IsBannedOnLegacyCF = 1 
	or rlu.IsBlacklisted = 1 
	or rlu.IsEnrolled = 0 
	or (select max(up.lastLogin) from #usersLastLogin up where up.userid = rlu.userid)  <= @eligibilityDateRange);

-- 7.) a.) now get the left over owners at zero in staging for the projects that made it into RewardSplit
DELETE rss
FROM [dbo].[RewardSplit_Staging] AS rss 
WHERE rss.ProjectID in (SELECT ProjectID FROM [dbo].[RewardSplit] with (nolock) WHERE Percentage = 1) AND rss.Percentage = 0;

/* STEP 8 */
-- 8.) Delete any user that is not eligible and has less than 100% split
DELETE rss
	FROM [dbo].[RewardSplit_Staging] AS rss
	LEFT JOIN [dbo].[RewardUser] AS rlu ON rss.UserID = rlu.UserID
	WHERE rss.Percentage < 1.00
	AND (rlu.IsBannedOnLegacyCF = 1 
	or rlu.IsBlacklisted = 1 
	or rlu.IsEnrolled = 0 
	or (select max(up.lastLogin) from #usersLastLogin up where up.userid = rlu.userid) <= @eligibilityDateRange);

-- now all that is left in the staging table are users that are eligible or have a zero percent
-- so step 6 will handle distributing any missing split percentage for the eligible users 
/* STEP 9 */
-- 9.) Process all users in source table who have between 1% AND 99% allocated, at this point after step 5 removed all NON eligible user there only eligible ones left
IF object_id('tempdb..#eligibleUsers') is not null
    DROP TABLE #eligibleUsers

CREATE TABLE #eligibleUsers(id int identity, project int , percentage decimal(3,2));
INSERT INTO #eligibleUsers (project, percentage)
SELECT distinct(x.projectID) AS projectid, sum(x.percentage) AS percentage FROM (
SELECT rss.projectID AS projectID, rss.userID AS userID, rss.Percentage AS percentage
  FROM [dbo].[RewardSplit_Staging] AS rss
  WHERE rss.Percentage < 1.0
) AS x
GROUP BY x.projectID


WHILE EXISTS(SELECT TOP 1 1 FROM #eligibleUsers)
BEGIN
    --Get row FROM cursor to process
    SELECT TOP 1 @projectIDStep9 = project FROM #eligibleUsers  
	
	BEGIN
		IF object_id('tempdb..#projectDistroCheck') is not null
		DROP TABLE #projectDistroCheck
		
		CREATE TABLE #projectDistroCheck(id int, projectid int , userid int, percentage decimal(3,2), ownerBool bit);
		INSERT INTO #projectDistroCheck (id, projectid, userid, percentage, ownerBool)
		SELECT rss.id, rss.projectid, rss.userid, rss.percentage, ownerBool =
		CASE 
			WHEN rlp.OwnerID = rss.UserID THEN 1
			ELSE 0
			END 
		FROM [dbo].[RewardSplit_Staging] AS rss 
		LEFT JOIN [dbo].RewardLegacyProject AS rlp ON rss.ProjectID = rlp.ID 
		WHERE rss.projectid = @projectIDStep9 
		--AND rss.Percentage < 1.0
		ORDER BY ownerBool desc, rss.Percentage desc

		SELECT @eligUsersStep9 = COUNT(userid) FROM #projectDistroCheck;
		
		SELECT @percentToDistro = (1 - SUM(percentage)) FROM #projectDistroCheck

		IF @percentToDistro != 0
			BEGIN
				-- change all percentages to ints to get remainders, the back to decimals
				SET @percentToDistroINT = @percentToDistro * 100;
				SET @splitToAddToUserINT = (@percentToDistroINT / @eligUsersStep9);
				SET @remainderStep9INT = (@percentToDistroINT % @eligUsersStep9);
				-- back to decimal
				SET @splitToAddToUser = (@splitToAddToUserINT * 0.01);
				SET @remainderStep9 = (@remainderStep9INT * 0.01);
				SET @remainderGiven = 0;
				IF @eligUsersStep9 > 1
					WHILE EXISTS(SELECT TOP 1 1 FROM #projectDistroCheck)
					BEGIN
						SELECT TOP 1 @projectDistroCheckID = id, @userIdStep9 = userId, @userGivenPercentageStep6 = percentage, @isOwner = ownerBool FROM #projectDistroCheck
						IF (SELECT ownerBool FROM #projectDistroCheck WHERE ownerBool = 1) IS NULL AND @remainderGiven = 0
							BEGIN
								IF @remainderGiven = 0
									BEGIN
										SET @userGivenPercentageStep6 = @userGivenPercentageStep6 + @splitToAddToUser + @remainderStep9;
										SET @remainderGiven = 1
									END
								ELSE
									SET @userGivenPercentageStep6 = @userGivenPercentageStep6 + @splitToAddToUser;
							END
						ELSE
							BEGIN
								IF @isOwner = 1
									BEGIN
										SET @userGivenPercentageStep6 = @userGivenPercentageStep6 + @splitToAddToUser + @remainderStep9;
										SET @remainderGiven = 1
									END
								ELSE
									SET @userGivenPercentageStep6 = @userGivenPercentageStep6 + @splitToAddToUser;
							END
						SET @splitToDistroStep9 = @userGivenPercentageStep6;
						IF @splitToDistroStep9 > 0
							INSERT INTO [dbo].[RewardSplit] (projectID, userID, percentage, DailyBudgetDate)  SELECT @projectIDStep9, @userIdStep9, @splitToDistroStep9, GETUTCDATE();
						DELETE FROM #projectDistroCheck WHERE id = @projectDistroCheckID
					END
				ELSE
					BEGIN
						SET @splitToDistroStep9 = 1.00 --there were no other eligible authors, so SET this user's % to 100
						SELECT @userIdStep9 = userid FROM #projectDistroCheck;
						INSERT INTO [dbo].[RewardSplit] (projectID, userID, percentage, DailyBudgetDate)  SELECT @projectIDStep9, @userIdStep9, @splitToDistroStep9, GETUTCDATE();
					END
			END
		ELSE
			BEGIN
				INSERT INTO [dbo].[RewardSplit] (projectid, userid, percentage, DailyBudgetDate)
				SELECT projectid, userid, percentage, GETUTCDATE() FROM #projectDistroCheck WHERE percentage > 0
			END

		DROP TABLE #projectDistroCheck
	END
	
    --Remove processed row FROM cursor. and from staging
    DELETE FROM #eligibleUsers WHERE project = @projectIDStep9;  
	DELETE FROM [dbo].[RewardSplit_Staging] WHERE ProjectID = @projectIDStep9;
END


DELETE s FROM [dbo].[RewardSplit_Staging] AS s
inner JOIN (
		SELECT distinct(rss.projectID) AS projectid
		FROM [dbo].[RewardSplit_Staging] AS rss
		WHERE rss.percentage > 0
		GROUP BY ProjectID)
  AS a ON s.projectid = a.ProjectID
  
DROP TABLE #eligibleUsers


/* STEP 10 */
-- 10.) Process all users in source table who have 0% allocated
INSERT INTO [dbo].[RewardSplit](projectid, userid, percentage, DailyBudgetDate) 
SELECT rss.projectid AS projectid, rlp.OwnerID AS userid, 1.00 AS percentage, GETUTCDATE()
FROM [dbo].[RewardSplit_Staging] AS rss
LEFT JOIN [dbo].[RewardLegacyProject] AS rlp ON rss.ProjectID = rlp.ID
LEFT JOIN [dbo].[RewardUser] AS rlu ON rlp.OwnerID = rlu.UserID
WHERE rlu.IsBannedOnLegacyCF = 0 
AND rlu.IsBlacklisted = 0
AND rlu.IsEnrolled = 1 
AND (select max(up.lastLogin) from #usersLastLogin up where up.userid = rlu.userid)  >= @eligibilityDateRange

DELETE FROM [dbo].[RewardSplit_Staging]
WHERE percentage = 0.0

DROP TABLE #usersLastLogin

COMMIT TRANSACTION;
END TRY

BEGIN CATCH
ROLLBACK TRANSACTION;
SELECT @errorMessage = ERROR_MESSAGE();
RETURN(920);
END CATCH
RETURN(1);
END


GO